/*
 * By KennethRoy
 */
(function($) {
    var Scroller = function(element, userSettings) {
        var $element,
            elements = {},
            settings,
            lastScrollY,
            lastScrollPercentage,
            observer;

        var defaultSettings = {
            root: null,
            offset: '0px',
            sensitivity: 0,
            classes: {
                scroller: 'scroller',
                scrollerActive: 'scroller-active',
            },
            callback: function() {},
            onWindowScroll: function() {},
            onWindowResize: function() {},
        };

        var initElements = function() {
            $element = $(element).addClass(settings.classes.scroller);

            elements.$window = $(window);
        };

        var initSettings = function() {
            settings = $.extend(true, defaultSettings, userSettings);
        };

        var bindEvents = function() {
            // elements.$window.on({
            //     scroll: onWindowScroll,
            //     resize: onWindowResize,
            // });
        };

        this.onWindowScroll = function() {
            this.offWindowScroll();
            elements.$window.on('scroll', settings.onWindowScroll);
        };

        this.offWindowScroll = function() {
            elements.$window.off('scroll', settings.onWindowScroll);
        };

        this.onWindowResize = function() {
            this.offWindowResize();
            elements.$window.on('resize', settings.onWindowResize);
        };

        this.offWindowResize = function() {
            elements.$window.off('resize', settings.onWindowResize);
        };

        var unbindEvents = function() {
            elements.$window
                .off('scroll', onWindowScroll)
                .off('resize', onWindowResize);
        };

        var init = function() {
            initSettings();

            initElements();

            bindEvents();

            const options = {
                root: settings.root || null,
                rootMargin: settings.offset || '0px',
                threshold: buildThresholds(settings.sensitivity)
            };

            observer = new IntersectionObserver(handleIntersect, options);
            observer.observe($element[0]);
        };

        this.destroy = function() {
            unbindEvents();

            observer.unobserve($element);

            $element.removeClass(settings.classes.scroller);
        };

        var onWindowScroll = function() {
            // settings.onWindowScroll();
        };

        var onWindowResize = function() {
            // settings.onWindowResize();
        };

        this.activate = function() {
            $element.addClass(settings.classes.scrollerActive);
        };

        this.deactivate = function() {
            $element.removeClass(settings.classes.scrollerActive);
        };

        var buildThresholds = function() {
            let sensitivityPercentage = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : 0;
            let thresholds = [];
            if ($.isArray(sensitivityPercentage)) {
                thresholds = sensitivityPercentage;
            } else {
                if (sensitivityPercentage > 0 && sensitivityPercentage <= 100) {
                    const increment = 100 / sensitivityPercentage;
                    for (let i = 0; i <= 100; i += increment) {
                        thresholds.push(i / 100);
                    }
                } else {
                    thresholds.push(0);
                }
            }

            return thresholds;
        };

        var handleIntersect = function(entries) {
            const currentScrollY = entries[0].boundingClientRect.y,
                isInViewport = entries[0].isIntersecting,
                intersectionScrollDirection = currentScrollY < lastScrollY ? 'down' : 'up',
                scrollPercentage = Math.abs(parseFloat((entries[0].intersectionRatio * 100).toFixed(2))),
                isDisappearing = scrollPercentage < lastScrollPercentage;
            settings.callback({
                $element: $element,
                sensitivity: settings.sensitivity,
                isInViewport,
                scrollPercentage,
                isDisappearing,
                intersectionScrollDirection
            });
            lastScrollY = currentScrollY;
            lastScrollPercentage = scrollPercentage;
        };

        var getElementViewportPercentage = function($element) {
            let offsetObj = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
            const elementOffset = $element[0].getBoundingClientRect(),
                offsetStart = offsetObj.start || 0,
                offsetEnd = offsetObj.end || 0,
                windowStartOffset = window.innerHeight * offsetStart / 100,
                windowEndOffset = window.innerHeight * offsetEnd / 100,
                y1 = elementOffset.top - window.innerHeight,
                y2 = elementOffset.top + windowStartOffset + $element.height(),
                startPosition = 0 - y1 + windowStartOffset,
                endPosition = y2 - y1 + windowEndOffset,
                percent = Math.max(0, Math.min(startPosition / endPosition, 1));
            return parseFloat((percent * 100).toFixed(2));
        }

        var getPageScrollPercentage = function() {
            let offsetObj = arguments.length > 0 && arguments[0] !== undefined ? arguments[0] : {};
            let limitPageHeight = arguments.length > 1 ? arguments[1] : undefined;
            const offsetStart = offsetObj.start || 0,
                offsetEnd = offsetObj.end || 0,
                initialPageHeight = limitPageHeight || document.documentElement.scrollHeight - document.documentElement.clientHeight,
                heightOffset = initialPageHeight * offsetStart / 100,
                pageRange = initialPageHeight + heightOffset + initialPageHeight * offsetEnd / 100,
                scrollPos = document.documentElement.scrollTop + document.body.scrollTop + heightOffset;
            return scrollPos / pageRange * 100;
        }

        init();
    };

    $.fn.scroller = function(settings) {
        var isCommand = 'string' === typeof settings;

        this.each(function() {
            var $this = $(this);

            if (!isCommand) {
                $this.data('scroller', new Scroller(this, settings));

                return;
            }

            var instance = $this.data('scroller');

            if (!instance) {
                throw Error('Trying to perform the `' + settings + '` method prior to initialization');
            }

            if (!instance[settings]) {
                throw ReferenceError('Method `' + settings + '` not found in scroller instance');
            }

            instance[settings].apply(instance, Array.prototype.slice.call(arguments, 1));

            if ('destroy' === settings) {
                $this.removeData('scroller');
            }
        });

        return this;
    };

    $.fn.getElementViewportPercentage = function() {
        let $element = $(this);
        let offsetObj = arguments.length > 1 && arguments[1] !== undefined ? arguments[1] : {};
        const elementOffset = $element[0].getBoundingClientRect(),
            offsetStart = offsetObj.start || 0,
            offsetEnd = offsetObj.end || 0,
            windowStartOffset = window.innerHeight * offsetStart / 100,
            windowEndOffset = window.innerHeight * offsetEnd / 100,
            y1 = elementOffset.top - window.innerHeight,
            y2 = elementOffset.top + windowStartOffset + $element.height(),
            startPosition = 0 - y1 + windowStartOffset,
            endPosition = y2 - y1 + windowEndOffset,
            percent = Math.max(0, Math.min(startPosition / endPosition, 1));
        return parseFloat((percent * 100).toFixed(2));
    };

    window.Scroller = Scroller;
})(jQuery);